:- module(bu,[]).

% bottom-up chart parser with both active and inactive items. This
% one uses packing, and a restriction in the first phase, to have
% fewer and smaller items. The second phase adds full constraints
% for top category spanning the whole wordgraph (if any) 
% or for all top categories with a partial span (otherwise)

count :-
	debug:debug_call(1,count_edges:
	  (  bu_chart:current_predicate(_,X),
	     report_count_edges(bu_chart:X),
	     fail
	  ;  true
	  )			
			).

count(X) :-
	count_edges:program_space(X).

clean :-
	(  bu_chart:current_predicate(_,X), 
	   bu_chart:retractall(X), 
	   fail
	;  true
	).


parse(o(Cat,Phon,_)):-
	P0 = 1,
	length(Phon,P1), P is P1+1,
	count_edges:reset_program_space,
	process_word_graph,
	user:result_cat(Cat,Syn),
	user:result_tree(Cat,Tree),
	bu_chart:second(P0,P,Syn,Ref),
	second_phase(Ref,Syn,Tree).


process_word_graph :-
	( %lexical_analysis:
          %    clause(lex_analysis(P0,P,LHS,_,_,Id,_,_),_,Ref),
	  %lex:clause(syn(P0,P,LHS),_,Ref),
	  lex:lex(P0,P,LHS,Ids),
	  store([],LHS,P0,P,leaves(Ids),[]),
	  fail
	; true
	).

store([],LHS0,P0,P,RuleId,His) :-
	weaken(LHS0,LHS),
	LHS =.. [F|Args],
	INACTIVE =.. [F,P0,P|Args],
	assertz_if_new(bu_chart:INACTIVE,Ref,New),
	bu_chart:assertz(his(Ref,RuleId,His)),
	continue_inactive(New,P0,P,LHS,Ref).

% is weakening useful here? At best to save space.
store([H0|T0],LHS0,P0,P,RuleId,His) :-
	weaken(LHS0,LHS),
	weaken_list([H0|T0],[H|T]),
	H =.. [F|Args],
	ACTIVE =.. [F,P,P0,LHS,T,RuleId,His|Args],
	assertz(bu_chart:ACTIVE).

continue_inactive(not_new,_,_,_,_).
continue_inactive(    new,P0,P,LHS,Ref) :-
	initiate(P0,P,LHS,Ref),
	extend(P0,P,LHS,Ref),
	add_for_second_phase(P0,P,LHS,Ref).

initiate(P0,P,First,Ref) :-
	(  bu_in:grammar_rule(First,Id,LHS,_,Rest),
	   store(Rest,LHS,P0,P,rule(Id),[Ref]),
	   fail
	;  true
	).

extend(P1,P,First,Ref) :-
	(  %% active(P1,P0,First,LHS,Rest,Id,Refs0),
	   First =.. [F|Args],
	   ACTIVE =.. [F,P1,P0,LHS,Rest,Id,Refs0|Args],
	   bu_chart:ACTIVE,
	   lists:append(Refs0,[Ref],Refs),
	   store(Rest,LHS,P0,P,Id,Refs),
	   fail
	;  true
	).

add_for_second_phase(P0,P,LHS,Ref) :-
	(  grammar:top_category(LHS),
	   bu_chart:assertz(second(P0,P,LHS,Ref)),
	   fail
	;  true
	).


second_phase(Ref,Cat,Tree) :-
	(  bu_chart:reconstructed(Ref)
	-> true
	;  bu_chart:assertz(reconstructed(Ref)),
	   (  second_phase0(Ref,Cat0,Tree0),
	      bu_chart:assertz(reconstructed(Ref,Cat0,Tree0)),
	      fail
	   ;  true
	   )
	),
	bu_chart:reconstructed(Ref,Cat,Tree).

second_phase0(Ref,Cat0,Tree0) :-
	bu_chart:his(Ref,Rule,Refs),
	second_phase2(Rule,Cat0,Refs,Tree0).

second_phase2(rule(Rule),Cat,Refs,tree(Rule,_,Trees)) :-
	mj_in:grammar_rule(Rule,_,Cat,Rhs),
	second_phase_list(Refs,Rhs,Trees).

second_phase2(leaves(Ids),Cat,[],tree(Id,_,[])) :-
	lists:member(Id,Ids),
        lex_in:total_lex(Id,Cat,_,_).
	%Ref = '$ref'(Refa,Refb),
	%lex:total(Refa,Refb,_,_,Cat,Id).
	% lexical_analysis:clause(lex_analysis(_,_,Cat,Cat,Sc1,_,_,Tr0),_,Ref),
	% lists:append(Tr0,Tail,Tr).

second_phase_list([],[],[]).
second_phase_list([H|T],[H2|T2],[Tr|Trs]) :-
	second_phase(H,H2,Tr),
	second_phase_list(T,T2,Trs).

compile_grammar(File) :-
	user:ensure_grammar_compiled_for_parser(act,File),
	bu_in:abolish(grammar_rule/5),
	(  mj_in:grammar_rule(Id,LHS,LHS2,[First|Rest]),
	   bu_in:assertz(grammar_rule(First,Id,LHS,LHS2,Rest)),
	   fail
	;  true
	).

dump_grammar:-
	user:dump_predicate(bu_in:grammar_rule(_,_,_,_,_)),
	user:dump_predicate(mj_in:grammar_rule(_,_,_,_)).

weaken_list([],[]).
weaken_list([H|T],[H2|T2]) :-
	weaken(H,H2),
	weaken_list(T,T2).    

%% :- flags:initialize_flag(restrict_bu,2).
weaken(Term0,Term) :-
	flags:flag(restrict_bu,Val),
	(  Val == undefined
	-> Term0=Term
	;  restrict:restrict(Term0,Val,Term)
	).


% this declaration is useless if this module is loaded from a .ql file
:- meta_predicate assertz_if_new(:,-,-).

% assertz Pred0 with reference Ref if no more general one exists
% in that case New is instantiated as new.
% otherwise Ref is the reference of a more general item.
assertz_if_new(Module:Pred0,Ref,New) :-
	(  more_general(Module:Pred0,Ref)    % more general exists
	-> New=not_new
	;  New=new,
	   Module:assertz(Pred0,Ref)
	).

:- meta_predicate more_general(:,?).
more_general(Module:Pred,Ref) :-
	copy_term(Pred,PredC),
	Module:clause(PredC,_,Ref),
	Module:clause(General,_,Ref),
	terms:subsumes_chk(General,Pred).

