/*  File:  hopf.c  */

/*  Discrete Hopfield Network
 *
 *  Encoding: Teuvo Kohonen
 *            Self-Organization and Associative Memory
 *            Third Edition
 *            Springer Verlag, Berlin, 1989
 *
 *            paragraaf 6.7.2
 *            vergelijking: 6.33 eerste deel
 *                          6.34
 */

#ifdef __MSDOS__
#ifndef __COMPACT__
#error Memory model COMPACT required
#endif  /* __COMPACT__ */
#include <conio.h>
#include <dir.h>
#endif  /* __MSDOS__ */

#include <ctype.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <values.h>

#define BUFSIZE 2048

#define PLUS '@'
#define MIN  '.'

enum BOOL { FALSE = 0, TRUE = 1 };

struct VEC
{
    float
        *x,
        *_x,
        _x_square;
    char
        *label;
}
    *vec = NULL;

float
    **w;

int
    *u,
    *o,
    n_vec = 0,
    max_vec = 0,
    noise = 0,
    breed,
    hoog,
    scr_width,
    n_units,
    input_line;

char
    **txt,
    *programname,
    buffer [BUFSIZE + 1],
    *no_mem_buffer,
    *patternfile = NULL,
    *weightfile = NULL,
    *seedfile = NULL,
    *testpattern = NULL,
    *savefile = "_weights.hop",
    *no_mem_buffer,
    Out_of_memory [] = "Out of memory!",
    file_error [] = "File \"%s\", line %i",
    file_not_found [] = "File \"%s\" not found",
    illegal_value [] = "Illegal value for %s",
    illegal_in_file [] = "Illegal value for %s in file \"%s\"",
    width [] = "width",
    height [] = "height";

enum BOOL
    mirror = FALSE,
    corr = FALSE;

void
    project_record (void),
    correlate_record (void),
    save_weights (void),
    get_testpattern (void),
    show (void),
    txt_flush (void),
    energy (void),
    get_programname (char const *argv0),
    process_args (int argc, char *argv []),
    setup (void),
    read_patternfile (void),
    read_weightfile (void),
    read_seedfile (void),
    syntax (void),
    errit (char const *format, ...),
    *s_malloc (size_t size),
    *s_realloc (void *block, size_t size);
char
    *s_strdup (char const *s),
    *getline (FILE *fp, enum BOOL required, char const *filename);
char const
    *get_arg (int argc, char *argv [], int *index);

int main (int argc, char *argv [])
{
    int
	i,
	j,
	n,
	x,
        y,
	next,
        swap;
    float
	som;
    enum BOOL
	bezig;

#ifdef __MSDOS__
    struct text_info
	ti;
    gettextinfo (&ti);
    scr_width = ti.screenwidth;
#else
    char
	*c;
    c = getenv ("COLUMNS");
    if (c) {
	scr_width  = atoi (c);
	if (scr_width < 1)
	    scr_width = 80;
    } else
	scr_width = 80;
#endif

    no_mem_buffer = (char *) malloc (1024);

    get_programname (argv [0]);

    process_args (argc, argv);

    srand ((unsigned int) time (NULL));

    if ((! weightfile) && ! patternfile)
	errit ("Missing weightfile or patternfile");

    if (weightfile)
	read_weightfile ();

    if (patternfile)
	read_patternfile ();

    if (! weightfile) {
	w = (float **) s_malloc (n_units * sizeof (float *));
	for (y = 0; y < n_units; y++) {
	    w [y] = (float *) s_malloc (n_units * sizeof (float));
	    for (x = 0; x < n_units; x++)
		w [y][x] = 0.0;
	}
	if (corr)
	    correlate_record ();
	else
	    project_record ();
	for (i = 0; i < n_units; i++)
	    w [i][i] = 0.0;
	save_weights ();
    }

    u = (int *) s_malloc (n_units * sizeof (int));

    if (testpattern) {
	if (! patternfile)
	    errit ("Missing patternfile for \"%s\"", testpattern);
	get_testpattern ();
    } else if (seedfile)
	read_seedfile ();
    else
	for (i = 0; i < n_units; i++)
            u [i] = -1;

    for (i = 0; i < n_units; i++)
	if ((rand () % 100) < noise)
            u [i] *= -1;

    if (mirror)
        u [n_units - 1] = 1;

    o = (int *) s_malloc (n_units * sizeof (int));
    for (i = 0; i < n_units; i++)
	o [i] = i;

    if (hoog > 1) {
        txt = (char **) s_malloc (hoog * sizeof (char *));
        for (i = 0; i < hoog; i++) {
	    txt [i] = (char *) s_malloc ((scr_width + 1) * sizeof (char));
	    txt [i][0] = '\0';
	}
    }

    putchar ('\n');
    energy ();

    show ();

    for (bezig = TRUE; bezig; ) {
	bezig = FALSE;

	for (i = n_units - 1; i > 0; i--) {
	    n = rand () % (i + 1);
	    swap = o [n];
	    o [n] = o [i];
	    o [i] = swap;
	}

	for (i = 0; i < n_units; i++) {
	    n = o [i];
            som = 0.0;
	    for (j = 0; j < n_units; j++)
                som += w [j][n] * (float) u [j];
            if (som > 0.0)
		next = 1;
            else if (som < 0.0)
		next = -1;
	    else
		next = u [n];
	    if (u [n] != next) {
		u [n] = next;
		bezig = TRUE;
                if (mirror && n == n_units - 1)
                    for (j = 0; j < n_units; j++)
                        u [j] *= -1;
		show ();
		break;
	    }
	}
    }

    if (hoog > 1 && txt [0][0])
	txt_flush ();

    if (hoog == 1)
        putchar ('\n');

    energy ();

    return 0;
}

void energy ()
{
    int
        i,
        j,
	n;
    float
        H;

    n = mirror ? (n_units - 1) : n_units;

    H = 0.0;
    for (i = 0; i < n; i++)
        for (j = 0; j < n; j++)
            H += w [i][j] * ((float) u [i]) * (float) u [j];
    H = -0.5 * H;
    printf ("%f\n\n", H);
}

void show ()
{
    int
	i,
	x,
	y;

    if (hoog == 1) {
        for (i = 0; i < breed; i++)
	    putchar ((u [i] > 0) ? PLUS : MIN);
        putchar ('\n');
    } else {
	if (strlen (txt [0]) + 2 + breed >= scr_width)
	    txt_flush ();
	if (txt [0][0])
            for (i = 0; i < hoog; i++)
		strcat (txt [i], "  ");
	for (y = 0; y < hoog; y++) {
	    i = strlen (txt [y]);
	    for (x = 0; x < breed; x++)
		txt [y][i++] = (u [y * breed + x] > 0) ? PLUS : MIN;
	    txt [y][i] = '\0';
	}
    }
}

void txt_flush ()
{
    int
	i;

    for (i = 0; i < hoog; i++) {
	puts (txt [i]);
	txt [i][0] = '\0';
    }
    putchar ('\n');
}
void read_weightfile ()
{
    int
        i,
        j,
        x,
	y;
    FILE
        *fp;

    fp = fopen (weightfile, "r");
    if (! fp)
        errit (file_not_found, weightfile);
    input_line = 0;

    getline (fp, TRUE, weightfile);
    if (sscanf (buffer, "%i %i %i %i", &n_units, &breed, &hoog, &i) != 4)
	errit (file_error, patternfile, input_line);
    mirror = (i ? TRUE : FALSE);

    w = (float **) s_malloc (n_units * sizeof (float *));
    for (i = 0; i < n_units; i++) {
	w [i] = (float *) s_malloc (n_units * sizeof (float));
	for (j = 0; j < n_units; j++)
	    w [i][j] = 0;
    }
    for (y = 0; y < n_units; y++)
        for (x = y; x < n_units; x++) {
	    getline (fp, TRUE, weightfile);
	    w [x][y] = w [y][x] = atof (buffer);
	}

    fclose (fp);
}

void read_patternfile ()
{
    int
	i,
	j,
	k;
    FILE
	*fp;

    fp = fopen (patternfile, "r");
    if (! fp)
        errit (file_not_found, patternfile);
    input_line = 0;

    getline (fp, TRUE, patternfile);
    if (sscanf (buffer, "%i %i", &i, &j) != 2)
	errit (file_error, patternfile, input_line);

    if (weightfile) {
	if (breed != i)
            errit (illegal_in_file, width, patternfile);
	if (hoog != j)
            errit (illegal_in_file, height, patternfile);
    } else {
	breed = i;
	hoog = j;
	if (breed < 1)
            errit (illegal_value, width);
	if (hoog < 1)
            errit (illegal_value, height);

	n_units = breed * hoog;

	if (mirror)
	    n_units++;
    }

    while (getline (fp, FALSE, NULL)) {
	while (n_vec >= max_vec) {
	    max_vec += 64;
	    vec = (struct VEC *) s_realloc (vec, max_vec * sizeof (struct VEC));
	}
	vec [n_vec].label = strdup (buffer);
	vec [n_vec].x  = (float *) s_malloc (n_units * sizeof (float));
	vec [n_vec]._x = (float *) s_malloc (n_units * sizeof (float));
	if (mirror)
	    vec [n_vec].x [n_units - 1] = 1.0;
	for (i = 0; i < hoog; i++) {
	    getline (fp, TRUE, patternfile);
	    j = 0;
	    k = -1;
	    while (j < breed) {
		switch (buffer [++k]) {
		    case ' ':
		    case '\t':
			break;
		    case '-':
		    case '0':
			vec [n_vec].x [i * breed + j++] = -1.0;
			break;
		    case '+':
		    case '1':
			vec [n_vec].x [i * breed + j++] = 1.0;
			break;
		    default:
			errit (
			    file_error,
			    patternfile,
			    input_line
			);
		}
	    }
	}
	n_vec++;
    }
    fclose (fp);
}

void read_seedfile ()
{
    int
	i,
	j,
	k;
    FILE
	*fp;

    fp = fopen (seedfile, "r");
    if (! fp)
        errit (file_not_found, seedfile);
    input_line = 0;

    for (i = 0; i < hoog; i++) {
	getline (fp, TRUE, seedfile);
	j = 0;
	k = -1;
	while (j < breed) {
	    switch (buffer [++k]) {
		case ' ':
		case '\t':
		    break;
		case '-':
		case '0':
		    u [i * breed + j++] = -1;
		    break;
		case '+':
		case '1':
		    u [i * breed + j++] = 1;
		    break;
		default:
		    errit (
			file_error,
			seedfile,
			input_line
		    );
	    }
	}
    }
    fclose (fp);
}

void get_testpattern ()
{
    int
	i,
	j;

    for (i = 0; i < n_vec; i++)
	if (! strcmp (vec [i].label, testpattern)) {
	    for (j = 0; j < n_units; j++)
		u [j] = vec [i].x [j];
	    return;
	}

    errit ("Testpattern \"%s\" not found", testpattern);
}

void project_record ()
{
    int
	i,
	j,
	k,
	x,
	y;
    float
	*xtmp,
	som;

    printf ("\nStoring patterns: Using Projection-Recording\n");

    xtmp = (float *) s_malloc (n_units * sizeof (float));
    for (i = 0; i < n_vec; i++) {

	printf ("Storing vector %s...", vec [i].label);
	fflush (stdout);

	/*  compute vec [i]._x  */
	memcpy (vec [i]._x, vec [i].x, n_units * sizeof (float));
	for (j = 0; j < i; j++) {
	    if (! vec [j]._x_square)
		continue;
	    som = 0.0;
	    for (k = 0; k < n_units; k++)
		som += vec [i].x [k] * vec [j]._x [k];
	    som /= vec [j]._x_square;
	    for (k = 0; k < n_units; k++)
		vec [i]._x [k] -= som * vec [j]._x [k];
	}

	/*  compute vec [i]._x_square  */
	som = 0.0;
	for (k = 0; k < n_units; k++)
	    som += vec [i]._x [k] * vec [i]._x [k];
	vec [i]._x_square = som;

	/*  compute new w  */
	if (! vec [i]._x_square) {
	    printf ("FAILED\n");
	    fflush (stdout);
	} else {
	    for (y = 0; y < n_units; y++) {
		som = 0.0;
		for (x = 0; x < n_units; x++)
		    som += w [y][x] * vec [i].x [x];
		xtmp [y] = som;
	    }

	    for (k = 0; k < n_units; k++)
		xtmp [k] = vec [i].x [k] - xtmp [k];

	    for (y = 0; y < n_units; y++)
		for (x = 0; x < n_units; x++)
		    w [y][x] += xtmp [y] * vec [i]._x [x] / vec [i]._x_square;

	    printf ("ok\n");
	    fflush (stdout);
	}
    }
    free (xtmp);
}

void correlate_record ()
{
    int
	i,
	x,
	y;

    printf ("\nStoring patterns: Using Correlation-Recording\n");

    for (i = 0; i < n_vec; i++) {
        putchar ('.');
        fflush (stdout);
	for (y = 0; y < n_units; y++)
	    for (x = 0; x < n_units; x++)
		w [y][x] += vec [i].x [y] * vec [i].x [x] / (float) n_units;
    }
    putchar ('\n');
}

void save_weights ()
{
    int
	x,
	y;
    FILE *
	fp;

    fp = fopen (savefile, "w");
    if (! fp)
	errit ("Unable to create file \"%s\"", savefile);
    fprintf (
	fp,
	"%i %i %i %i\n",
	n_units,
	breed,
	hoog,
	(int)(mirror ? 1 : 0)
    );
    for (y = 0; y < n_units; y++)
	for (x = y; x < n_units; x++)
	    fprintf (fp, "%f\n", w [y][x]);
    fclose (fp);
    printf ("\nWeights saved in file \"%s\"\n", savefile);
}

char *getline (FILE *fp, enum BOOL required, char const *filename)
{
    int
	i;

    for (;;) {
	if (fgets (buffer, BUFSIZE, fp) == NULL) {
	    if (required)
		errit ("Unexpected end of file in \"%s\"", filename);
	    else
		return NULL;
	}
	input_line++;
	i = strlen (buffer);
	while (i && isspace (buffer [i - 1]))
	    buffer [--i] = '\0';
	i = 0;
	while (buffer [i] && isspace (buffer [i]))
	    i++;
	if (buffer [i] == '#')
	    continue;
	if (buffer [i]) {
	    memmove (buffer, buffer + i, strlen (buffer) + 1);
	    return buffer;
	}
    }
}

void syntax ()
{
    fprintf (
	stderr,
        "\nDiscrete Hopfield Network\n\n"
        "(c) P. Kleiweg 1996\n\n"
	"Syntax: %s -p string | -w string [-s string | -t string]\n"
	"\t\t[-n int] [-c] [-m]\n\n"
	"-p : patternfile\n"
	"-w : weightfile\n"
	"-s : seedfile\n"
	"-t : label of testpattern\n"
	"-n : random noise percentage: 0 - 100\n"
	"-c : use correlation-recording\n"
	"-m : no mirror images\n\n",
	programname
    );
    exit (1);
}

void process_args (int argc, char *argv [])
{
    int
	i;

    if (argc == 1)
	syntax ();

    for (i = 1; i < argc; i++) {
	if (argv [i][0] != '-' && argv [i][0] != '/')
	    errit ("Illegal argument \"%s\"", argv [i]);
	switch (argv [i][1]) {
	    case 'p':
		patternfile = s_strdup (get_arg (argc, argv, &i));
		break;
	    case 'w':
		weightfile = s_strdup (get_arg (argc, argv, &i));
		break;
	    case 's':
		seedfile = s_strdup (get_arg (argc, argv, &i));
		break;
	    case 't':
		testpattern = s_strdup (get_arg (argc, argv, &i));
		break;
	    case 'n':
		noise = atoi (get_arg (argc, argv, &i));
		if (noise < 0 || noise > 100)
                    errit (illegal_value, "noise");
		break;
	    case 'c':
		corr = TRUE;
		break;
	    case 'm':
		mirror = TRUE;
		break;
	    default:
		errit ("Illegal option '%s'", argv [i]);
	}
    }
}

char const *get_arg (int argc, char *argv [], int *index)
{
    if (argv [*index][2])
	return argv [*index] + 2;

    if (*index == argc - 1)
        errit ("Argument missing for '%s'", argv [*index]);

    return argv [++*index];
}

void get_programname (char const *argv0)
{
#ifdef __MSDOS__
    char
	name [MAXFILE];
    fnsplit (argv0, NULL, NULL, name, NULL);
    programname = s_strdup (name);
#else   /* unix */
    char
	*p;
    p = strrchr (argv0, '/');
    if (p)
	programname = s_strdup (p + 1);
    else
	programname = s_strdup (argv0);
#endif
}

void *s_malloc (size_t size)
{
    void
	*p;

    p = malloc (size);
    if (! p) {
	free (no_mem_buffer);
	errit (Out_of_memory);
    }
    return p;
}

void *s_realloc (void *block, size_t size)
{
    void
	*p;

    p = realloc (block, size);
    if (! p) {
	free (no_mem_buffer);
	errit (Out_of_memory);
    }
    return p;
}

char *s_strdup (char const *s)
{
    char
	*s1;

    if (s) {
	s1 = (char *) s_malloc (strlen (s) + 1);
	strcpy (s1, s);
    } else {
	s1 = (char *) s_malloc (1);
	s1 [0] = '\0';
    }
    return s1;
}

void errit (char const *format, ...)
{
    va_list
	list;

    fprintf (stderr, "\nError %s: ", programname);

    va_start (list, format);
    vfprintf (stderr, format, list);

    fprintf (stderr, "\n\n");

    exit (1);
}
