/* File bp.c
 *
 * Back Propagation
 */

#ifdef __MSDOS__
#include <dir.h>
#endif

#include <ctype.h>
#include <math.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <values.h>

struct IO_PAIR
{
    char
	*name;
    float
        *v_in,
        *v_out;
}
    *io_pairs = NULL;

#define BUFSIZE 2048

enum BOOL { FALSE = 0, TRUE = 1 };

char
    *programname,
    buffer  [BUFSIZE + 1],	/* voor algemeen tijdelijk gebruik */
    buffer2 [BUFSIZE + 1],      /* voor gebruik door split_buffer() */
    *outbase = "bp_out",        /* basename voor uitvoer-bestanden */
    *net_file = NULL,           /* naam file met definitie van netwerk */
    *pat_file = NULL,           /* naam van file met input/output paren */
    *wt_file = NULL,            /* naam van file met oude gewichten */
    current_in [BUFSIZE],
    *no_mem_buffer,
    Out_of_memory [] = "Out of memory";

int
    n_patterns = 0,             /* aantal patronen */
    max_patterns = 0,           /* plaats in tabel voor patronen */
    n_units,                    /* aantal units */
    n_in,                       /* aantal invoerunits (de eerste units) */
    n_hid,                      /* aantal hidden units  */
    n_out,                      /* aantal uitvoerunits (de laatste units) */
    inA,                        /* de eerste invoerunit     */
    inZ,                        /* de laatste invoerunit    */
    hidA,                       /* de eerste hidden unit    */
    hidZ,                       /* de laatste hidden unit   */
    outA,                       /* de eerste uitvoerunit    */
    outZ,                       /* de laatste uitvoerunit   */
    n_epochs = 1000,		/* aantal epochs */
    log_step = 5,
    verbose = 1,
    input_line;

enum BOOL
    ordered = TRUE,             /* aanbieding trainingsparen geordend */
    **connected;                /* connected [i][j]: verbinding van unit [j]
                                   naar unit [i]  */

float
    eta = 0.35,                 /* leerconstante    */
    alfa = 0.9,                 /* momentum         */
    init_weight = 1.0,          /* maximale beginwaarde gewichten   */
    quit_error = 0.05,          /* maximale fout voor succes        */
    *unit,      /* de bias, input, hidden en output units */
    *d,         /* errors units  */
    *tar,       /* tar[0]..tar [n_out - 1]  */
    **w,        /* w [i][j] is gewicht van unit [j] naar unit [i] */
    **odw;      /* oude aanpassing w */

void
    get_programname (char const *argv0),
    process_args (int argc, char *argv []),
    setup (void),
    save_files (void),
    read_files (void),
    net_check (void),
    syntax (void),
    *s_malloc (size_t size),
    *s_realloc (void *block, size_t size),
    get_pattern (int),
    trim (char *s),
    feed_forward (void),
    back_propagate (void),
    split_buffer (char *dest, char *src),  	/*  src wordt ingekort  */
    no_mem (void),
    errit (char const *format, ...),
    Heapsort (		/* een veilige sorteermethode */
	void *base,
	size_t nelem,
	size_t width,
	int (*fcmp)(void *, void *)
    );
char
    *get_arg (int argc, char *argv [], int *index),
    *s_strdup (char const *s);
int
    getline (char *dest, FILE *fp, enum BOOL required);
float
    **alloc_matrix_float (int i1, int i2, float (*init)(void)),
    *alloc_vector_float (int i1, float (*init)(void)),
    init_random (void),
    get_square_error (void);
FILE
    *r_fopen (char const *filename),
    *w_fopen (char const *filename);

int main (int argc, char *argv [])
{
    char
        c [] = "-\\|/";
    int
        cc = 0,
	i,
        worst,
	epoch,
	pattern;
    float
        sum,
	max,
	err;
    FILE
        *fp;

    no_mem_buffer = (char *) malloc (1024);

    get_programname (argv [0]);

    process_args (argc, argv);

    read_files ();

    setup ();

    /* laat zien wat er gaat gebeuren */
    if (verbose) {
        printf (
            "Eta:       %f\n"
            "Alfa:      %f\n",
            (float) eta,
            (float) alfa
        );
        if (wt_file)
            printf (
                "Gewichten: vanuit bestand \"%s\"\n",
                (char *) wt_file
            );
        else
            printf (
                "Gewichten: [-%f, %f]\n",
            (float) init_weight,
            (float) init_weight
        );
        printf (
            "Succes:    error < %f\n"
            "Epochs:    %i\n"
            "Selectie trainingsparen: %s\n",
            (float) quit_error,
            (int) n_epochs,
            ordered ? "geordend" : "willekeurig"
        );
    }

    sprintf (buffer, "%s.log", outbase);
    fp = w_fopen (buffer);
    fprintf (
        fp,
        "Netwerk:   %s\n"
        "Patronen:  %s\n"
        "Eta:       %f\n"
        "Alfa:      %f\n",
        (char *) net_file,
        (char *) pat_file,
	(float) eta,
        (float) alfa
    );
    if (wt_file)
        fprintf (
            fp,
            "Gewichten: %s\n",
            (char *) wt_file
        );
    else
        fprintf (
            fp,
            "Gewichten: [-%f, %f]\n",
            (float) init_weight,
            (float) init_weight
        );
    fprintf (
        fp,
	"Selectie trainingsparen: %s\n",
	ordered ? "geordend" : "willekeurig"
    );
    fclose (fp);

    sprintf (buffer, "%s.out", outbase);
    fp = w_fopen (buffer);
    for (epoch = 1; epoch <= n_epochs; epoch++) {

        if (verbose == 2) {
            if (++cc >= 4)
                cc = 0;
            printf ("\r%c", c [cc]);
            fflush (stdout);
        }

	/* trainen van het netwerk */
	for (pattern = 0; pattern < n_patterns; pattern++) {
	    get_pattern (ordered ? pattern : (rand () % n_patterns));
	    feed_forward ();
	    back_propagate ();
	}
	if (epoch % log_step)
	    continue;

	/* huidige fitness van het netwerk bepalen */
        max = sum = 0.0;
	for (pattern = 0; pattern < n_patterns; pattern++) {
	    get_pattern (pattern);
	    feed_forward ();

	    err = sqrt (get_square_error ());
            sum += err;
            if (err > max) {
		max = err;
                worst = pattern;
            }

	    /* huidige performance van netwerk naar log_file */
	    fprintf (
                fp,
                "%s %i %f",
		(char *) io_pairs [pattern].name,
                (int) epoch,
                (float) err
	    );
            for (i = outA; i <= outZ; i++)
                fprintf (fp, " %f", unit [i]);
            fprintf (fp, "\n");
	}
        fprintf (
            fp,
            "_MEAN_\t%i\t%f\n"
            "_WORST_\t%i\t%f\n",
            (int) epoch,
            (float) sum / (float) n_patterns,
            (int) epoch,
            (float) max
        );

	/* toon huidig: epoch, som van fouten, grootste fout */
        if (verbose) {
            printf (
                " %i\t%f\t%s: %f\r",
                (int) epoch,
                (float) sum / (float) n_patterns,
                (char *) io_pairs [worst].name,
                (float) max
            );
            fflush (stdout);
        }

	if (max < quit_error)
	    break;
    }
    if (verbose)
        printf ("\n");
    fclose (fp);

    /* bewaar het getrainde netwerk */
    save_files ();

    return 0;
}

void get_pattern (int p)
{
    memcpy (unit + inA,  io_pairs [p].v_in,  n_in  * sizeof (float));
    memcpy (tar, io_pairs [p].v_out, n_out * sizeof (float));
}

void feed_forward ()
{
    int
	i, j;
    float
        sum;

    for (i = hidA; i <= outZ; i++) {
        sum = 0.0;
        for (j = 0; j < i; j++)
            if (connected [i][j])
                sum += w [i][j] * unit [j];
        unit [i] = 1.0 / (1.0 + exp (-sum));
    }
}

void back_propagate ()
{
    int
	i,
	j;
    float
        sum;

    for (i = outZ; i >= outA; i--)
        d [i] = unit [i] * (1.0 - unit [i]) * (tar [i - outA] - unit [i]);

    for (i = hidZ; i >= hidA; i--) {
        sum = 0.0;
        for (j = i + 1; j <= outZ; j++)
            if (connected [j][i])
                sum += d [j] * w [j][i];
        d [i] = unit [i] * (1.0 - unit [i]) * sum;
    }

    for (j = 0; j < n_units; j++)
        for (i = j + 1; i <= n_units; i++)
            if (connected [i][j])
                w [i][j] +=
                    (odw [i][j] = eta * d [i] * unit [j]
                                + alfa * odw [i][j]);
}

float get_square_error ()
{
    int
	i;
    float
	err,
        sum = 0.0;

    for (i = 0; i < n_out; i++) {
        err = unit [outA + i] - tar [i];
        sum += err * err;
    }
    return sum;
}

void setup ()
{
    FILE
        *fp;
    int
        i,
        j;

    srand ((unsigned int) time (NULL));

    unit = alloc_vector_float (n_units + 1, NULL);
    unit [0] = 1.0;

    d    = alloc_vector_float (n_units + 1, NULL);
    tar  = alloc_vector_float (n_out,   NULL);

    w    = alloc_matrix_float (n_units + 1, n_units + 1, init_random);
    odw  = alloc_matrix_float (n_units + 1, n_units + 1, NULL);


    /* oude gewichten inlezen */
    if (wt_file) {
        fp = r_fopen (wt_file);
        for (j = 0; j < n_units; j++)
            for (i = j + 1; i <= n_units; i++) {
                getline (buffer, fp, TRUE);
                if (sscanf (buffer, "%f", &(w [i][j])) != 1)
                    errit (
                        "Reading file \"%s\", line %i",
                        wt_file,
                        input_line
                    );
            }
        fclose (fp);
    }
}

/* genereer een beginwaarde voor een gewicht
   tussen -init_weight en init_weight		*/
float init_random ()
{
    return (init_weight > 0.0)
      ? (((float) rand () / RAND_MAX) * init_weight * 2.0 - init_weight)
      : 0.0;
}

void read_files ()
{
    int
	i,
	j;
    char
	*file_error = "Fout in bestand \"%s\", regel %i",
	*illegal = "Ongeldige waarde voor %s";
    FILE
        *fp;

    /* configuratie van het netwerk */

    if (! net_file)
        errit ("Geen naam voor netwerk-bestand gegeven");

    fp = r_fopen (net_file);
    getline (buffer, fp, TRUE);
    if (sscanf (buffer, "%i %i %i", &n_units, &n_in, &n_out) != 3)
        errit (file_error, net_file, input_line);
    n_hid = n_units - n_in - n_out;
    if (n_in < 1)
        errit (illegal, "n_in");
    if (n_out < 1)
	errit (illegal, "n_out");
    if (n_hid < 0)
	errit (illegal, "n_units");

    inA = 1;
    inZ = n_in;
    hidA = inZ + 1;
    hidZ = inZ + n_hid;
    outA = hidZ + 1;
    outZ = n_units;

    connected = (enum BOOL **)
        s_malloc ((n_units + 1) * sizeof (enum BOOL *));
    for (i = 0; i <= n_units; i++) {
        connected [i] = (enum BOOL *)
            s_malloc ((n_units + 1) * sizeof (enum BOOL));
	for (j = 0; j <= n_units; j++)
            connected [i][j] = FALSE;
    }

    while (getline (buffer, fp, FALSE)) {
	split_buffer (buffer2, buffer);
        j = atoi (buffer2);
        if (j < 0 || j > n_units)
            errit (file_error, net_file, input_line);
	split_buffer (buffer2, buffer);
	if (strcmp (buffer2, "->"))
	    errit (file_error, net_file, input_line);
	while (buffer [0]) {
	    split_buffer (buffer2, buffer);
            i = atoi (buffer2);
            if (i < 0 || i > n_units)
		errit (file_error, net_file, input_line);
            if (connected [i][j])
                errit ("Dubbele verbinding van unit %i naar unit %i", j, i);
            connected [i][j] = TRUE;
	}
    }
    fclose (fp);
    net_check ();

    /* de invoer/uitvoer patronen */

    if (! pat_file)
	errit ("Geen naam voor patterns-bestand gegeven");
    fp = r_fopen (pat_file);
    while (getline (buffer, fp, FALSE)) {
	while (n_patterns >= max_patterns) {
	    max_patterns += 64;
	    io_pairs = (struct IO_PAIR *)
                s_realloc (io_pairs, max_patterns * sizeof (struct IO_PAIR));
	}
	io_pairs [n_patterns].v_in  = (float *)
            s_malloc (n_in  * sizeof (float));
	io_pairs [n_patterns].v_out = (float *)
            s_malloc (n_out * sizeof (float));

	split_buffer (buffer2, buffer);
        if (! (buffer2 [0] >= 'a' && buffer2 [0] <= 'z' ||
               buffer2 [0] >= 'A' && buffer2 [0] <= 'Z' ||
               buffer2 [0] == '_')
        )
            errit ("Label ontbreekt, bestand \"%s\", regel %i", pat_file, input_line);
        for (i = 0; i < n_patterns; i++)
            if (! strcmp (buffer2, io_pairs [i].name))
                errit ("Twee patronen met dezelfde naam \"%s\"", buffer2);
        io_pairs [n_patterns].name = s_strdup (buffer2);
	for (i = 0; i < n_in; i++) {
	    split_buffer (buffer2, buffer);
	    io_pairs [n_patterns].v_in [i] = atof (buffer2);
	}
	for (i = 0; i < n_out; i++) {
	    split_buffer (buffer2, buffer);
	    io_pairs [n_patterns].v_out [i] = atof (buffer2);
	}
        n_patterns++;
    }
    fclose (fp);
}

void net_check ()
{
    int
        i,
        j,
        k;

    for (j = 0; j <= n_units; j++)
        for (i = 0; i <= n_units; i++)
            if (connected [i][j]) {
                if (j > i)
                    errit (
                        "Verbinding naar unit met lager nummer: %i -> %i",
                        j,
                        i
                    );
                if (j == i)
                    errit ("Unit met verbinding naar zichzelf: %i", j);
                if (j == 0 && i <= inA)
                    errit ("Bias naar invoerunit: %i -> %i", j, i);
                if (i <= inA)
                    errit ("Verbindig naar invoerunit: %i -> %i", j, i);
                if (j >= outA)
                    errit ("Verbinding vanuit uitvoerunit: %i -> %i", j, i);
            }

    for (j = inA; j <= inZ; j++) {
        k = 0;
        for (i = hidA; i <= outZ; i++)
            if (connected [i][j]) {
                k = 1;
                break;
            }
        if (! k)
            errit ("Geen verbinding vanuit invoerunit: %i", j);
    }
    for (i = outA; i <= outZ; i++) {
        k = 0;
        for (j = inA; j <= hidZ; j++)
            if (connected [i][j]) {
                k = 1;
                break;
            }
        if (! k)
            errit ("Geen verbinding naar uitvoerunit: %i", i);
    }
    for (j = hidA; j <= hidZ; j++) {
        k = 0;
        for (i = j + 1; i <= outZ; i++)
            if (connected [i][j]) {
                k = 1;
                break;
            }
        if (! k)
            errit ("Geen verbinding vanuit hidden unit: %i", j);
    }
    for (i = hidA; i <= hidZ; i++) {
        k = 0;
        for (j = inA; j < i; j++)
            if (connected [i][j]) {
                k = 1;
                break;
            }
        if (! k)
            errit ("Geen verbinding naar hidden unit: %i", i);
    }
}

void syntax ()
{
    fprintf
    (
	stderr,
        "\nBackpropagation Network\n"
        "\n(c) P. Kleiweg 1996\n"
        "\nSyntax: %s -n string -p string [-g string]\n"
	"\t[-e float] [-a float] [-w float]\n"
	"\t[-o string]\n"
        "\t[-m int] [-q float] [-s int] [-r] [-z | -v]\n\n"
	"-n : naam netwerkfile\n"
	"-p : naam bestand met patterns\n"
        "-g : naam bestand met gewichten, bij eerdere sessie bewaard\n"
	"-e : eta (leerconstante)\n"
	"-a : alfa (momentum)\n"
	"-w : maximale beginwaarde gewichten\n"
	"-o : naam van outputfiles zonder extensie\n"
	"-m : aantal epochs\n"
	"-q : quit wanneer grootste fout onder deze waarde komt\n"
	"-s : log step voor uitvoer naar file\n"
        "-r : random selectie van trainingsparen\n"
        "-z : stil\n"
        "-v : verbose\n\n",
	programname
    );
    exit (1);
}

void process_args (int argc, char *argv [])
{
    int
	i;
    char
	*illegal = "Ongeldige waarde voor %s";

    if (argc == 1)
        syntax ();

    for (i = 1; i < argc; i++) {
	if (argv [i][0] != '-')
            errit ("Illegal argument");
	switch (argv [i][1]) {
	    case 'n':
                net_file = s_strdup (get_arg (argc, argv, &i));
		break;
	    case 'p':
                pat_file = s_strdup (get_arg (argc, argv, &i));
		break;
            case 'g':
                wt_file = s_strdup (get_arg (argc, argv, &i));
		break;
	    case 'e':
		eta = atof (get_arg (argc, argv, &i));
		if (eta <= 0.0)
		    errit (illegal, "eta");
		break;
	    case 'a':
		alfa = atof (get_arg (argc, argv, &i));
		if (alfa < 0.0 || alfa >= 1.0)
		    errit (illegal, "alfa");
		break;
	    case 'w':
		init_weight = atof (get_arg (argc, argv, &i));
		if (init_weight < 0.0)
		    errit (illegal, "beginwaarde gewichten");
		break;
	    case 'o':
                outbase = s_strdup (get_arg (argc, argv, &i));
		break;
	    case 'm':
		n_epochs = atoi (get_arg (argc, argv, &i));
		break;
	    case 'q':
		quit_error = atof (get_arg (argc, argv, &i));
		break;
	    case 's':
		log_step = atoi (get_arg (argc, argv, &i));
		if (log_step <= 0)
		    errit (illegal, "log step");
		break;
	    case 'r':
		ordered = FALSE;
		break;
            case 'z':
                verbose = 0;
		break;
            case 'v':
                verbose = 2;
		break;
	    default:
                errit ("Illegal option '%s'", argv [i]);
	}
    }
}

char *get_arg (int argc, char *argv [], int *index)
{
    if (argv [*index][2])
	return argv [*index] + 2;

    if (*index == argc - 1)
	errit ("Argument voor optie '%s' ontbreekt", argv [*index]);

    return argv [++*index];
}

void save_files ()
{
    int
	i,
	j;
    FILE
	*fp;

    /* gewichten */
    sprintf (buffer, "%s.wt", outbase);
    fp = w_fopen (buffer);
    for (j = 0; j < n_units; j++) {
        fprintf (
            fp,
            "# unit [%i] -> unit [%i..%i]\n",
            j,
            j + 1,
            n_units
        );
        for (i = j + 1; i <= n_units; i++)
            fprintf (
                fp,
                "\t%f\n",
                (float) (connected [i][j] ? w [i][j] : 0.0)
            );
    }
    fclose (fp);

    /* bewaar het netwerk voor gnuplot */
    sprintf (buffer, "%s.fun", outbase);
    fp = w_fopen (buffer);
    fprintf (fp, "sigmoid (x) = 1.0 / (1.0 + exp (-x))\n");
    strcpy (buffer, "i1");
    for (i = 2; i <= n_in; i++) {
	sprintf (buffer2, ",i%i", (int) i);
	strcat (buffer, buffer2);
    }
    for (i = hidA; i <= outZ; i++) {
	fprintf (
	    fp,
	    "unit%i(%s)=sigmoid(%f",
            (int) i,
	    (char *) buffer,
            (float) (connected [i][0] ? w [i][0] : 0.0)
	);
        for (j = inA; j <= inZ; j++)
            if (connected [i][j])
		fprintf (
		    fp,
                    "\\\n +%f*i%i",
                    (float) w [i][j],
                    (int) j
		);
        for (j = hidA; j < i; j++)
            if (connected [i][j])
		fprintf (
		    fp,
                    "\\\n +%f*unit%i(%s)",
                    (float) w[i][j],
                    (int) j,
		    (char *) buffer
		);
	fprintf (fp, ")\n");
    }
    fclose (fp);
}

int getline (char *dest, FILE *fp, enum BOOL required)
{
    for (;;) {
	if (fgets (dest, BUFSIZE, fp) == NULL) {
	    if (required)
		errit ("Unexpected end of file in \"%s\"", current_in);
	    else
		return 0;
	}
	input_line++;
	trim (dest);
	if (dest [0])
	    return 1;
    }
}

/*  src wordt ingekort  */
void split_buffer (char *dest, char *src)
{
    if (sscanf (src, "%s", dest) != 1)
	errit (
	    "Voortijdig eind van string, file \"%s\", line %i",
	    (char *) current_in,
	    (int) input_line
	);
    memmove (src, src + strlen (dest), strlen (src) + 1);
    trim (dest);
    trim (src);
}

void trim (char *s)
{
    int
	i;

    /*  commentaar verwijderen  */
    for (i = 0; s [i]; i++)
	if (s [i] == '#' && (i == 0 || isspace (s [i - 1]))) {
	    s [i] = '\0';
	    break;
	}

    /*  spaties aan het einde verwijderen  */
    for (i = strlen (s) - 1; i >= 0; i--)
	if (isspace (s [i]))
            s [i] = '\0';
	else
	    break;

    /*  voorloopspaties verwijderen  */
    for (i = 0; s [i] && isspace (s [i]); i++)
	;
    memmove (s, s + i, strlen (s) - i + 1);
}

FILE *r_fopen (char const *filename)
{
    FILE
	*fp;

    fp = fopen (filename, "r");
    if (! fp)
	errit ("Bestand \"%s\" niet gevonden", filename);

    input_line = 0;
    strcpy (current_in, filename);
    return fp;
}

FILE *w_fopen (char const *filename)
{
    FILE
	*fp;

    fp = fopen (filename, "w");
    if (! fp)
        errit ("Bestand \"%s\" kan niet gemaakt worden", filename);
    return fp;
}

float **alloc_matrix_float (int i1, int i2, float (*init)(void))
{
    int
	i,
	j;
    float
	**f;

    f = (float **) s_malloc (i1 * sizeof (float *));
    for (i = 0; i < i1; i++) {
        f [i] = (float *) s_malloc (i2 * sizeof (float));
        for (j = 0; j < i2; j++)
            f [i][j] = init ? init () : 0.0;
    }
    return f;
}

float *alloc_vector_float (int i1, float (*init)(void))
{
    int
        i;
    float
        *f;

    f = (float *) s_malloc (i1 * sizeof (float));
    for (i = 0; i < i1; i++)
        f [i] = init ? init () : 0.0;

    return f;
}

void *s_malloc (size_t size)
{
    void
	*p;

    p = malloc (size);
    if (! p) {
        free (no_mem_buffer);
	errit (Out_of_memory);
    }
    return p;
}

void *s_realloc (void *block, size_t size)
{
    void
	*p;

    p = realloc (block, size);
    if (! p) {
        free (no_mem_buffer);
	errit (Out_of_memory);
    }
    return p;
}

char *s_strdup (char const *s)
{
    char
	*s1;

    if (s) {
        s1 = (char *) s_malloc (strlen (s) + 1);
	strcpy (s1, s);
    } else {
        s1 = (char *) s_malloc (1);
	s1 [0] = '\0';
    }
    return s1;
}

void get_programname (char const *argv0)
{
#ifdef __MSDOS__
    char
        name [MAXFILE];
    fnsplit (argv0, NULL, NULL, name, NULL);
    programname = strdup (name);
#else   /* unix */
    char
        *p;
    p = strrchr (argv0, '/');
    if (p)
        programname = strdup (p + 1);
    else
        programname = strdup (argv0);
#endif    
}

/* foutmelding, daarna exit */
void errit (char const *format, ...)
{
    va_list
	list;

    fprintf (stderr, "\nError %s: ", programname);

    va_start (list, format);
    vfprintf (stderr, format, list);

    fprintf (stderr, "\n\n");

    exit (1);
}

/* gebruik identiek aan gebruik qsort(), maar dit is veiliger */
void Heapsort
    (
	void *base,
	size_t nelem,
	size_t width,
	int (*fcmp)(void *, void *)
    )
{
    unsigned
	L, R, S, J;
    char
	*A,
	*heapswap;

    if (nelem < 2)
	return;

    A = (char *) base;
    heapswap = (char *) s_malloc (width);

    A -= width;                 /* table: A[1]..        */
    R = nelem;                  /*              ..A[R]  */
    L = (R >> 1) + 1;
    while (R > 1) {

	if (L > 1)
	    /* generate reverse partial ordered tree */
	    --L;
	else {
	    /* delete max from reverse partial ordered list
	       and append in front of ordered list    */
	    memcpy (heapswap, A + width, width);
	    memcpy (A + width, A + R * width, width);
	    memcpy (A + R-- * width, heapswap, width);
	}

	/* A[L+1]..A[R] is reverse partial ordered tree
	   insert A[L] to make A[L]..A[R] reverse partial ordered tree  */
	memcpy (heapswap, A + L * width, width);
	J = L;
	for (;;) {
	    S = J;              /* mother         */
	    J <<= 1;            /* left daughter  */
	    if (J > R)          /* no daughters   */
		break;
	    if ( (J < R) && (fcmp (A + J * width, A + (J + 1) * width) < 0) )
		J++;            /*  right daughter  */
	    if (fcmp (heapswap, A + J * width) >= 0)
		break;
	    memcpy (A + S * width, A + J * width, width);
	}
	memcpy (A + S * width, heapswap, width);
    }

    free (heapswap);
}
